require 'puppet/parameter/boolean'
require_relative '../../pacemaker/options'
require_relative '../../pacemaker/type'

Puppet::Type.newtype(:pacemaker_nodes) do
  desc <<-eof
Add and remove cluster nodes using the "corosync-cmapctl"
tool without restart of the Pacemaker and Corosync services.
  eof

  include Pacemaker::Options
  include Pacemaker::Type

  newparam(:name) do
    isnamevar
  end

  newparam(:debug, boolean: true, parent: Puppet::Parameter::Boolean) do
    desc <<-eof
Don't actually make changes

Default: false
eof
    defaultto false
  end

  newparam(:nodes) do
    desc <<-eos
Nodes data structure:
  ```
  {
    '1' => {
      name: 'node1',
      id: '1',
      ring0: '192.168.0.1',
      ring1: '172.16.0.1',
    },
    '2' => {
      name: 'node2',
      id: '2',
      ring0: '192.168.0.2',
      ring1: '172.16.0.2',
    },
    '3' => {
      name: 'node3',
      id: '3',
      ring0: '192.168.0.3',
      ring1: '172.16.0.3',
    }
  }
  ```
This structure may be generated by the "corosync_nodes"
function with "raw" output type from the several input formats.
    eos

    def filter_nodes_structure(nodes)
      filtered_nodes = {}
      node_attributes = %w(id name ring0 ring1)
      nodes.each do |node_id, node_hash|
        node = {}
        node_attributes.each do |node_attribute|
          node[node_attribute] = node_hash[node_attribute] if node_hash[node_attribute]
        end
        filtered_nodes[node_id] = node
      end
      filtered_nodes
    end

    def insync?(should)
      debug "#{is.inspect} vs #{should.inspect}"
      filter_nodes_structure(is) == filter_nodes_structure(should)
    end

    def validate(value)
      fail 'Nodes should be a non-empty hash!' unless value.is_a? Hash and value.any?
    end
  end

  newparam(:remove_pacemaker_nodes) do
    defaultto true
  end

  autorequire(:service) do
    %w(corosync pacemaker pcsd)
  end

end
