# Copyright (c) 2011 OpenStack, LLC.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
"""
Utilization Based Weighers.  Weighs hosts based on actual resource utilization.

Weighers providing load balancing and server consolidation according to the
currently most critical resource for achieving the objective, cpu or memory.
"""

from nova.openstack.common import cfg
from nova.scheduler import weights


utilization_weight_opts = [
        cfg.FloatOpt('load_balancing_weight_multiplier',
                     default=1.0,
                     help='Multiplier used for load balancing weighing.'),
        cfg.FloatOpt('server_consolidation_weight_multiplier',
                     default=1.0,
                     help='Multiplier used for server consolidation weighing.')
]

CONF = cfg.CONF
CONF.register_opts(utilization_weight_opts)


def _get_normalized_cpu_usage(host_state, weight_properties):
    instance_type = weighing_properties.get('instance_type')
    cpu_mhz = host_state.cpu_mhz_total / host_state.vcpus
    requested_cpu_mhz = instance_type['vcpus'] * cpu_mhz
    used_cpu_mhz = host_state.cpu_mhz_used
    cpu_mhz_limit = host_state.limits.get('cpu_mhz_usage',
                                          host_state.cpu_mhz_total)
    return (requested_cpu_mhz + used_cpu_mhz) / float(cpu_mhz_limit)


def _get_normalized_ram_usage(host_state, weight_properties):
    instance_type = weighing_properties.get('instance_type')
    requested_ram_mb = instance_type['memory_mb']
    used_ram_mb = host_state.memory_mb_used
    ram_mb_limit = host_state.limits.get('memory_mb_usage',
                                         host_state.total_usable_ram_mb)
    return (requested_ram_mb + used_ram_mb) / float(ram_mb_limit)


class LoadBalancingWeigher(weights.BaseHostWeigher):
    def _weight_multiplier(self):
        """Override the weight multiplier."""
        return CONF.load_balancing_weight_multiplier

    def _weigh_object(self, host_state, weight_properties):
        """Weight calculated from most used resource."""
        cpu_usage = _get_normalized_cpu_usage(host_state, weight_properties)
        ram_usage = _get_normalized_ram_usage(host_state, weight_properties)
        return 1 - max(cpu_usage, ram_usage)


class ServerConsolidationWeigher(weights.BaseHostWeigher):
    def _weight_multiplier(self):
        """Override the weight multiplier."""
        return CONF.server_consolidation_weight_multiplier

    def _weigh_object(self, host_state, weight_properties):
        """Weight calculated from least used resource."""
        cpu_usage = _get_normalized_cpu_usage(host_state, weight_properties)
        ram_usage = _get_normalized_ram_usage(host_state, weight_properties)
        # An empty host always has the least possible weight
        if host_state.num_instances == 0:
            return 0
        return min(cpu_usage, ram_usage)
