# Copyright 2014 IBM Corp.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from oslo_config import cfg
from oslo_config import types


"""Configurations for reference implementation

I think that these configuration parameters should have been split into
small groups and be set into each implementation where they get used.

For example: kafka configuration should have been in the implementation
where kafka get used. It seems to me that the configuration for kafka gets
used in kafka_publisher, but the original settings were at the api/server.py
which I think is at the wrong place. I move these settings here for now, we
need to have a bit more re-engineering to get it right.
"""
global_opts = [cfg.StrOpt('region', help='Region that API is running in')]

cfg.CONF.register_opts(global_opts)


security_opts = [cfg.ListOpt('default_authorized_roles', default=['admin'],
                             help='Roles that are allowed full access to the '
                                  'API'),
                 cfg.ListOpt('agent_authorized_roles', default=['agent'],
                             help='Roles that are only allowed to POST to '
                                  'the API'),
                 cfg.ListOpt('delegate_authorized_roles', default=['admin'],
                             help='Roles that are allowed to POST metrics on '
                                  'behalf of another tenant')]

security_group = cfg.OptGroup(name='security', title='security')
cfg.CONF.register_group(security_group)
cfg.CONF.register_opts(security_opts, security_group)

messaging_opts = [cfg.StrOpt('driver', default='kafka',
                             help='The message queue driver to use'),
                  cfg.StrOpt('events_message_format', default='reference',
                             help='The type of events message format to '
                                  'publish to the message queue')]

messaging_group = cfg.OptGroup(name='messaging', title='messaging')
cfg.CONF.register_group(messaging_group)
cfg.CONF.register_opts(messaging_opts, messaging_group)

repositories_opts = [
    cfg.StrOpt('streams',
               default='monasca_events_api.common.repositories.streams_repository:StreamsRepository',
               help='The repository driver to use for streams'),
    cfg.StrOpt('events',
               default='monasca_events_api.common.repositories.events_repository:EventsRepository',
               help='The repository driver to use for events'),
    cfg.StrOpt('transforms',
               default='monasca_events_api.common.repositories.transforms_repository:TransformsRepository',
               help='The repository driver to use for transforms')]

repositories_group = cfg.OptGroup(name='repositories', title='repositories')
cfg.CONF.register_group(repositories_group)
cfg.CONF.register_opts(repositories_opts, repositories_group)


kafka_opts = [cfg.StrOpt('uri', help='Address to kafka server. For example: '
                                     'uri=192.168.1.191:9092'),
              cfg.StrOpt('events_topic', default='raw-events',
                         help='The topic that events will be published too.'),
              cfg.StrOpt('group', default='api',
                         help='The group name that this service belongs to.'),
              cfg.IntOpt('wait_time', default=1,
                         help='The wait time when no messages on kafka '
                              'queue.'), cfg.IntOpt('ack_time', default=20,
                                                    help='The ack time back '
                                                         'to kafka.'),
              cfg.IntOpt('max_retry', default=3,
                         help='The number of retry when there is a '
                              'connection error.'),
              cfg.BoolOpt('auto_commit', default=False,
                          help='If automatically commmit when consume '
                               'messages.'),
              cfg.BoolOpt('async', default=True, help='The type of posting.'),
              cfg.BoolOpt('compact', default=True, help=(
                  'Specify if the message received should be parsed.'
                  'If True, message will not be parsed, otherwise '
                  'messages will be parsed.')),
              cfg.MultiOpt('partitions', item_type=types.Integer(),
                           default=[0],
                           help='The sleep time when no messages on kafka '
                                'queue.'),
              cfg.BoolOpt('drop_data', default=False, help=(
                  'Specify if received data should be simply dropped. '
                  'This parameter is only for testing purposes.')), ]

kafka_group = cfg.OptGroup(name='kafka', title='title')
cfg.CONF.register_group(kafka_group)
cfg.CONF.register_opts(kafka_opts, kafka_group)

mysql_opts = [cfg.StrOpt('database_name'), cfg.StrOpt('hostname'),
              cfg.StrOpt('username'), cfg.StrOpt('password')]

mysql_group = cfg.OptGroup(name='mysql', title='mysql')
cfg.CONF.register_group(mysql_group)
cfg.CONF.register_opts(mysql_opts, mysql_group)
