/*
 * Decompiled with CFR 0.152.
 */
package monasca.persister.repository;

import com.codahale.metrics.Meter;
import com.codahale.metrics.Timer;
import com.google.inject.Inject;
import io.dropwizard.setup.Environment;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import monasca.persister.configuration.MonPersisterConfiguration;
import monasca.persister.repository.InfluxRepository;
import monasca.persister.repository.MetricRepository;
import monasca.persister.repository.Sha1HashId;
import org.apache.commons.codec.digest.DigestUtils;
import org.influxdb.dto.Serie;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class InfluxDBMetricRepository
extends InfluxRepository
implements MetricRepository {
    private static final Logger logger = LoggerFactory.getLogger(InfluxDBMetricRepository.class);
    private static final int NUMBER_COLUMNS = 2;
    private final Map<Sha1HashId, Def> defMap = new HashMap<Sha1HashId, Def>();
    private final Map<Sha1HashId, Set<Dim>> dimMap = new HashMap<Sha1HashId, Set<Dim>>();
    private final Map<Sha1HashId, DefDim> defDimMap = new HashMap<Sha1HashId, DefDim>();
    private final Map<Sha1HashId, List<Measurement>> measurementMap = new HashMap<Sha1HashId, List<Measurement>>();
    private final Timer flushTimer;
    public final Meter measurementMeter;
    private final SimpleDateFormat measurementTimeSimpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss zzz");
    private static final Sha1HashId BLANK_SHA_1_HASH_ID = new Sha1HashId(DigestUtils.sha((String)""));
    private static final Set<Dim> EMPTY_DIM_TREE_SET = new TreeSet<Dim>();

    @Inject
    public InfluxDBMetricRepository(MonPersisterConfiguration configuration, Environment environment) {
        super(configuration, environment);
        this.flushTimer = this.environment.metrics().timer(this.getClass().getName() + "." + "flush-timer");
        this.measurementMeter = this.environment.metrics().meter(this.getClass().getName() + "." + "measurement-meter");
    }

    @Override
    public void addMetricToBatch(Sha1HashId defDimsId, String timeStamp, double value) {
        Measurement measurement = new Measurement(defDimsId, timeStamp, value);
        List<Measurement> measurementList = this.measurementMap.get(defDimsId);
        if (measurementList == null) {
            measurementList = new LinkedList<Measurement>();
            this.measurementMap.put(defDimsId, measurementList);
        }
        measurementList.add(measurement);
    }

    @Override
    public void addDefinitionToBatch(Sha1HashId defId, String name, String tenantId, String region) {
        Def def = new Def(defId, name, tenantId, region);
        this.defMap.put(defId, def);
    }

    @Override
    public void addDimensionToBatch(Sha1HashId dimSetId, String name, String value) {
        Set<Dim> dimSet = this.dimMap.get(dimSetId);
        if (dimSet == null) {
            dimSet = new TreeSet<Dim>();
            this.dimMap.put(dimSetId, dimSet);
        }
        Dim dim = new Dim(dimSetId, name, value);
        dimSet.add(dim);
    }

    @Override
    public void addDefinitionDimensionToBatch(Sha1HashId defDimsId, Sha1HashId defId, Sha1HashId dimId) {
        DefDim defDim = new DefDim(defDimsId, defId, dimId);
        this.defDimMap.put(defDimsId, defDim);
    }

    @Override
    public void flush() {
        try {
            long startTime = System.currentTimeMillis();
            Timer.Context context = this.flushTimer.time();
            this.influxDB.write(this.configuration.getInfluxDBConfiguration().getName(), TimeUnit.SECONDS, this.getSeries());
            long endTime = System.currentTimeMillis();
            context.stop();
            logger.debug("Writing measurements, definitions, and dimensions to InfluxDB took {} seconds", (Object)((endTime - startTime) / 1000L));
        }
        catch (Exception e) {
            logger.error("Failed to write measurements to InfluxDB", (Throwable)e);
        }
        this.clearBuffers();
    }

    private String buildSerieName(Def def, Set<Dim> dimList) throws UnsupportedEncodingException {
        logger.debug("Creating serie name");
        StringBuilder serieNameBuilder = new StringBuilder();
        logger.debug("Adding tenant_id to serie name: {}", (Object)def.tenantId);
        serieNameBuilder.append(this.urlEncodeUTF8(def.tenantId));
        serieNameBuilder.append("?");
        logger.debug("Adding region to serie name: {}", (Object)def.region);
        serieNameBuilder.append(this.urlEncodeUTF8(def.region));
        serieNameBuilder.append("&");
        logger.debug("Adding name to serie name: {}", (Object)def.name);
        serieNameBuilder.append(this.urlEncodeUTF8(def.name));
        for (Dim dim : dimList) {
            serieNameBuilder.append("&");
            logger.debug("Adding dimension name to serie name: {}", (Object)dim.name);
            serieNameBuilder.append(this.urlEncodeUTF8(dim.name));
            serieNameBuilder.append("=");
            logger.debug("Adding dimension value to serie name: {}", (Object)dim.value);
            serieNameBuilder.append(this.urlEncodeUTF8(dim.value));
        }
        String serieName = serieNameBuilder.toString();
        logger.debug("Created serie name: {}", (Object)serieName);
        return serieName;
    }

    private Def getDef(Sha1HashId defId) throws Exception {
        Def def = this.defMap.get(defId);
        if (def == null) {
            throw new Exception("Failed to find definition for defId: " + defId);
        }
        return def;
    }

    private Set<Dim> getDimSet(Sha1HashId dimId) throws Exception {
        if (dimId.equals(BLANK_SHA_1_HASH_ID)) {
            return EMPTY_DIM_TREE_SET;
        }
        Set<Dim> dimSet = this.dimMap.get(dimId);
        if (dimSet == null) {
            throw new Exception("Failed to find dimension set for dimId: " + dimId);
        }
        return dimSet;
    }

    private String urlEncodeUTF8(String s) throws UnsupportedEncodingException {
        return URLEncoder.encode(s, "UTF-8");
    }

    private String[] buildColNamesStringArry() {
        String[] colNameStringArry = new String[2];
        colNameStringArry[0] = "time";
        logger.debug("Added column name[{}] = {}", (Object)0, (Object)colNameStringArry[0]);
        colNameStringArry[1] = "value";
        logger.debug("Added column name[{}] = {}", (Object)1, (Object)colNameStringArry[1]);
        if (logger.isDebugEnabled()) {
            this.logColumnNames(colNameStringArry);
        }
        return colNameStringArry;
    }

    private Serie[] getSeries() throws Exception {
        LinkedList<Serie> serieList = new LinkedList<Serie>();
        for (Sha1HashId defDimId : this.measurementMap.keySet()) {
            DefDim defDim = this.defDimMap.get(defDimId);
            Def def = this.getDef(defDim.defId);
            Set<Dim> dimSet = this.getDimSet(defDim.dimId);
            Serie.Builder builder = new Serie.Builder(this.buildSerieName(def, dimSet));
            builder.columns(this.buildColNamesStringArry());
            for (Measurement measurement : this.measurementMap.get(defDimId)) {
                Object[] colValsObjArry = new Object[2];
                Date date = this.measurementTimeSimpleDateFormat.parse(measurement.time + " UTC");
                Long time = date.getTime() / 1000L;
                colValsObjArry[0] = time;
                logger.debug("Added column value to colValsObjArry[{}] = {}", (Object)0, colValsObjArry[0]);
                colValsObjArry[1] = measurement.value;
                logger.debug("Added column value to colValsObjArry[{}] = {}", (Object)1, colValsObjArry[1]);
                builder.values(colValsObjArry);
                this.measurementMeter.mark();
            }
            Serie serie = builder.build();
            if (logger.isDebugEnabled()) {
                this.logColValues(serie);
            }
            serieList.add(serie);
            logger.debug("Added serie: {} to serieList", (Object)serie.getName());
        }
        return serieList.toArray(new Serie[serieList.size()]);
    }

    private void clearBuffers() {
        this.measurementMap.clear();
        this.defMap.clear();
        this.dimMap.clear();
        this.defDimMap.clear();
    }

    private static final class DefDim {
        final Sha1HashId defDimId;
        final Sha1HashId defId;
        final Sha1HashId dimId;

        private DefDim(Sha1HashId defDimId, Sha1HashId defId, Sha1HashId dimId) {
            this.defDimId = defDimId;
            this.defId = defId;
            this.dimId = dimId;
        }

        public String toString() {
            return "DefinitionDimension{defDimId=" + this.defDimId + ", defId=" + this.defId + ", " + "dimId=" + this.dimId + '}';
        }
    }

    private static final class Dim
    implements Comparable<Dim> {
        final Sha1HashId dimSetId;
        final String name;
        final String value;

        private Dim(Sha1HashId dimSetId, String name, String value) {
            this.dimSetId = dimSetId;
            this.name = name;
            this.value = value;
        }

        public String toString() {
            return "Dimension{dimSetId=" + this.dimSetId + ", name='" + this.name + '\'' + ", " + "value='" + this.value + '\'' + '}';
        }

        @Override
        public int compareTo(Dim o) {
            int nameCmp = String.CASE_INSENSITIVE_ORDER.compare(this.name, o.name);
            return nameCmp != 0 ? nameCmp : String.CASE_INSENSITIVE_ORDER.compare(this.value, o.value);
        }
    }

    private static final class Def {
        final Sha1HashId defId;
        final String name;
        final String tenantId;
        final String region;

        private Def(Sha1HashId defId, String name, String tenantId, String region) {
            this.defId = defId;
            this.name = name;
            this.tenantId = tenantId;
            this.region = region;
        }

        public String toString() {
            return "Definition{defId=" + this.defId + ", name='" + this.name + '\'' + ", " + "tenantId='" + this.tenantId + '\'' + ", region='" + this.region + '\'' + '}';
        }
    }

    private static final class Measurement {
        final Sha1HashId defDimsId;
        final String time;
        final double value;

        private Measurement(Sha1HashId defDimsId, String time, double value) {
            this.defDimsId = defDimsId;
            this.time = time;
            this.value = value;
        }

        public String toString() {
            return "Measurement{defDimsId=" + this.defDimsId + ", time='" + this.time + '\'' + ", " + "value=" + this.value + '}';
        }
    }
}

