# Copyright 2018 Red Hat, Inc.
# All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import os
import urllib

from tenacity import retry
from tenacity import retry_if_exception_type
from tenacity import stop_after_attempt
from tenacity import wait_exponential

from config_tempest import constants as C
from config_tempest.services.base import VersionedService
import json


class LoadBalancerService(VersionedService):
    def set_versions(self):
        super(LoadBalancerService, self).set_versions(top_level=False)

    def set_default_tempest_options(self, conf):
        conf.set('load_balancer', 'enable_security_groups', 'True')
        conf.set('load_balancer', 'admin_role', 'admin')
        conf.set('load_balancer', 'RBAC_test_type', 'owner_or_admin')
        conf.set('network-feature-enabled', 'port_security', 'True')
        # TOOD(gthiemonge) This is a backward-compatible setting for jobs
        # that haven't yet migrated to load_balancer.test_server_remote_url
        # Remove it once all jobs have been migrated.
        conf.set('load_balancer', 'test_server_path',
                 C.DEFAULT_OCTAVIA_COMPAT_TEST_SERVER_FILE)

    @staticmethod
    def get_service_type():
        return ['load-balancer']

    @staticmethod
    def get_codename():
        return 'octavia'

    def list_drivers(self):
        """List lbaas drivers"""
        body = self.do_get(self.service_url + '/v2/lbaas/providers')
        body = json.loads(body)
        names = [
            '{p[name]}:{p[description]}'.format(p=i) for i in body['providers']
        ]
        return names

    def post_configuration(self, conf, is_service):
        if not conf.has_option('auth', 'tempest_roles') \
                or conf.get('auth', 'tempest_roles') in ['', None]:
            conf.set('load_balancer', 'member_role', 'member')
        else:
            conf.set('load_balancer', 'member_role',
                     conf.get('auth', 'tempest_roles').split(',')[0])
        conf.set('load_balancer', 'region', conf.get('identity', 'region'))
        conf.set('load_balancer',
                 'enabled_provider_drivers',
                 ','.join(self.list_drivers()))

    @retry(retry=retry_if_exception_type(urllib.error.URLError),
           stop=stop_after_attempt(3),
           wait=wait_exponential(multiplier=2, min=3, max=10))
    def _download_file(self, url, destination):
        """Downloads a file specified by `url` to `destination`.

        :type url: string
        :type destination: string
        """
        if os.path.exists(destination):
            C.LOG.info("File '%s' already fetched to '%s'.", url, destination)
            return
        C.LOG.info("Downloading '%s' and saving as '%s'", url, destination)
        f = urllib.request.urlopen(url)
        data = f.read()
        with open(destination, "wb") as dest:
            dest.write(data)

    def get_test_server_application(self, conf):
        if not conf.has_option("load_balancer",
                               "test_server_remote_url"):
            return

        remote_url = conf.get("load_balancer",
                              "test_server_remote_url")
        if remote_url:
            self._download_file(remote_url,
                                C.DEFAULT_OCTAVIA_TEST_SERVER_FILE)
            os.chmod(C.DEFAULT_OCTAVIA_TEST_SERVER_FILE, 0o755)

            conf.set("load_balancer",
                     "test_server_path",
                     C.DEFAULT_OCTAVIA_TEST_SERVER_FILE)
